/*
 *  Ned full screen routines.
 */

#include <sgtty.h>
#define COI 0
#define MAXCOL 80
#define MAXLINE 24
#define SCRSIZE (MAXCOL * MAXLINE)
#define BUFSIZE 3000
#define NULL 0

/* AID chars */
#define ENTER 0x7d
#define TSTRQ 0xf0
#define PF1   0xf1
#define PF2   0xf2
#define PF3   0xf3
#define PF4   0xf4
#define PF5   0xf5
#define PF6   0xf6
#define PF7   0xf7
#define PF8   0xf8
#define PF9   0xf9
#define PF10  0x7a
#define PF11  0x7b
#define PF12  0x7c
#define PF13  0xc1
#define PF14  0xc2
#define PF15  0xc3
#define PF16  0xc4
#define PF17  0xc5
#define PF18  0xc6
#define PF19  0xc7
#define PF20  0xc8
#define PF21  0xc9
#define PF22  0x4a
#define PF23  0x4b
#define PF24  0x4c
#define PA1   0x6c
#define PA2   0x6e
#define PA3   0x6b
#define CLEAR 0x6d

/*
 * hex constants for 3270 control chars
 */
#define ERW 0x05
#define SF  0x1d
#define IC  0x13
#define SBA 0x11
#define SOH 0x01
#define STX 0x02

/* global declarations */

static char fsbuf[BUFSIZE],*fsbufp;
static char *fspath = 0;

int  fskp, fsrow, fscol;
static int fsfd = -1;
static int fsnread;
static char fslastcmd;

extern char tubecc[], tubotab[], mrxotab[], tubitab[];
extern char *cmdname;
extern int K;

/*
 * Open full screen tube.
 */
fsopen()
{
	char *ttyname();

	if (fsfd >= 0)
		return;
	if (fspath==0) {
		fspath = ttyname(2);
		if (fspath == 0) {
			printf("%s: standard error is not a terminal.\n", cmdname);
			exit(2);
		}
		/*
		 * Change tty name to tube name.
		 *    0123456789
		 *    /dev/ttyxx
		 *    /dev/tubxx
		 */
		fspath[6] = 'u';
		fspath[7] = 'b';
	}
	fsfd = open(fspath,2);
	if (fsfd < 0){
		printf("%s: cannot open %s\n", cmdname, fspath);
		exit(2);
	}
	fslastcmd = ERW;    /* Ensured by tube driver on open. */
}

/*
 * Close full screen device.
 */
fsclose()
{
	close(fsfd);
	fsfd = -1;
}

/*
 * Initialize buffer.
 */
fsbfinit(wcc)
int wcc;
{
	fsbufp = fsbuf;
	*fsbufp++ = wcc;
}

/*
 * Place cursor on screen.
 */
fscursor(rowcol)
int rowcol;
{
	*fsbufp++ = SBA;
	if (rowcol > SCRSIZE)
		rowcol = 0;
	*fsbufp++ = tubecc[rowcol >> 6];
	*fsbufp++ = tubecc[rowcol & 077];
	*fsbufp++ = IC;
}

/*
 * Put a field on the screen.
 * This is one of the real CPU burners in ned, hence
 * efficiency is more important than simplicity.
 */
fsputfld(cp, row, col, len, attr)
register char    *cp;
int     row, col;
register int     len;
int     attr;
{
	register int i;
	register char *fsp;
	register char *table;

	fsp = fsbufp;
	*fsp++ = SBA;
	i = row * MAXCOL + col - 1;
	if (i < 0)
		i += SCRSIZE;
	*fsp++ = tubecc[i >> 6];
	*fsp++ = tubecc[i & 077];
	*fsp++ = SF;
	*fsp++ = tubecc[attr];
	if (K == COI)
		table = tubotab;
	else
		table = mrxotab;
	/*
	 * This first loop is easier to read but the compiler
	 * generates less code for the other.
	 *
	 * for (i = 0; i < len && *cp; i++)
	 *         *fsp++ = table[*cp++];
	 */
	i = len;
	while (i && *cp) {
		len = table + *cp++;
		*fsp++ = * (char *) len;
		i--;
	}
	fsbufp = fsp;
}

/*
 * Write the buffer to the screen.
 */
fswrite(cmd)
register int cmd;
{
	register int n;

	if (fsfd < 0)
		fsopen();
	if (cmd != fslastcmd)
		ioctl(fsfd, TUBOCMD, fslastcmd = cmd);
	n = write(fsfd, fsbuf, fsbufp - fsbuf);
	if (n != fsbufp - fsbuf)
		nederr("Full screen write error.");
}

fsread()
{
	register int rowcol;

	fsnread = read(fsfd, fsbuf, BUFSIZE);
	if (fsnread <= 0)
		nederr("Full screen read error.");
	switch (fsbuf[0]) {           /* AID char */
		case ENTER:
			fskp=0;
			break;

		case PF1:  case PF2:  case PF3:  case PF4:  case PF5:
		case PF6:  case PF7:  case PF8:  case PF9:  case PF10:
		case PF11: case PF12:
			fskp = fsbuf[0] & 017;
			break;

		case PF13: case PF14: case PF15: case PF16: case PF17:
		case PF18: case PF19: case PF20: case PF21: case PF22:
		case PF23: case PF24:
			fskp = (fsbuf[0] & 017) + 12;
			break;

		case PA1:
			fskp=25;
			break;

		case PA2:
			fskp=26;
			break;

		case PA3:
			fskp=27;
			break;

		case CLEAR:
			fskp=28;
			break;

		case SOH:            /* TEST REQ */
			/* check for test req */
			if (fsbuf[1]==tubotab['%'] && fsbuf[2]==tubotab['/'] && fsbuf[3]==STX){
				fskp=29;
				break;
			}

		default:
			fskp = 100;
			break;
	}
	if (fskp <= 24) {
		fsbufp = &fsbuf[3];
		rowcol = ((fsbuf[1] & 077)<<6) + (fsbuf[2] & 077);
		fsrow = rowcol / MAXCOL;
		fscol = rowcol % MAXCOL;
	} else
		fsbufp = &fsbuf[4];
}

/*
 * Get a field from the read-in buffer.
 * Returns NULL if field was not read in (i.e. not modified).
 */
char *
fsgetfld(row, col)
int     row, col;
{
	register char *fsp;
	int rowcol;
	register char sba1, sba2;
	static char buf[512];
	register char *bufp, *fsend;

	rowcol = row * MAXCOL + col;
	sba1 = tubecc[rowcol>>6];
	sba2 = tubecc[rowcol & 077];

	fsp = fsbufp;
	fsend = fsbuf + fsnread;
	while (fsp < fsend) {
		if (*fsp++ == SBA && *fsp++ == sba1 && *fsp++==sba2) {
			bufp = buf;
			while (fsp < fsend && *fsp != SBA)
				*bufp++ = tubitab[*fsp++];
			*bufp = '\0';
			return(buf);
		}
	}
	return(NULL);
}
